param([switch]$Run)

$Preview = -not $Run
$Desktop = [Environment]::GetFolderPath("Desktop")

# ZIP name (ASCII)
$Today = Get-Date -Format "yyyyMMdd"
$ZipName = "DesktopFilesBackup_$Today.zip"
$ZipPath = Join-Path $Desktop $ZipName
if (Test-Path -LiteralPath $ZipPath) {
  $ts = Get-Date -Format "HHmmss"
  $ZipName = "DesktopFilesBackup_${Today}_$ts.zip"
  $ZipPath = Join-Path $Desktop $ZipName
}

$SkipLogName = "DesktopFilesBackup_${Today}_skipped.txt"
$SkipLogPath = Join-Path $Desktop $SkipLogName
if (Test-Path -LiteralPath $SkipLogPath) { Remove-Item -LiteralPath $SkipLogPath -Force -ErrorAction SilentlyContinue }

$RootName = "_BY_EXT"
$Root = Join-Path $Desktop $RootName

$ExcludeExactNames = @(
  $RootName,
  "DesktopSortByExt.ps1",
  "DesktopSortByExt_real.ps1",
  "sort_preview.bat",
  "sort_run.bat",
  $ZipName,
  $SkipLogName
)

function Ensure-Folder([string]$path) {
  if (-not (Test-Path -LiteralPath $path)) {
    if ($Preview) { Write-Host "[PREVIEW] Would create folder: $path" }
    else { New-Item -ItemType Directory -Path $path | Out-Null }
  }
}

function Log-Skip([string]$msg) {
  Add-Content -LiteralPath $SkipLogPath -Value $msg -Encoding UTF8
}

function Backup-DesktopFiles-ToZipDirect {
  if ($Preview) {
    Write-Host "[PREVIEW] Would create ZIP (FILES ONLY): $ZipPath"
    Write-Host "[PREVIEW] Skipped log: $SkipLogPath"
    return
  }

  # Prepare zip (overwrite)
  if (Test-Path -LiteralPath $ZipPath) { Remove-Item -LiteralPath $ZipPath -Force -ErrorAction SilentlyContinue }

  Add-Type -AssemblyName System.IO.Compression
  Add-Type -AssemblyName System.IO.Compression.FileSystem

  $files = Get-ChildItem -LiteralPath $Desktop -Force | Where-Object {
    -not $_.PSIsContainer -and ($ExcludeExactNames -notcontains $_.Name)
  }

  $added = 0
  $skipped = 0

  $zip = [System.IO.Compression.ZipFile]::Open($ZipPath, [System.IO.Compression.ZipArchiveMode]::Create)
  try {
    foreach ($f in $files) {
      try {
        # entry name = just filename (no paths)
        [System.IO.Compression.ZipFileExtensions]::CreateEntryFromFile($zip, $f.FullName, $f.Name, [System.IO.Compression.CompressionLevel]::Optimal) | Out-Null
        $added++
      } catch {
        $skipped++
        Log-Skip ("[SKIP] " + $f.FullName + " :: " + $_.Exception.Message)
      }
    }
  } finally {
    $zip.Dispose()
  }

  if (-not (Test-Path -LiteralPath $ZipPath)) {
    throw "ZIP not created: $ZipPath"
  }

  $size = (Get-Item -LiteralPath $ZipPath).Length
  Write-Host "ZIP backup created (FILES ONLY): $ZipPath"
  Write-Host "Backup summary: added=$added, skipped=$skipped, zipSizeBytes=$size"
  if ($skipped -gt 0) {
    Write-Host "Some files were skipped. See: $SkipLogPath"
  }
}

function Get-FolderNameFromExt([string]$ext) {
  if ([string]::IsNullOrWhiteSpace($ext)) { return "_NO_EXT" }
  return $ext.TrimStart('.').ToUpperInvariant()
}

function Move-FileSafely([string]$src, [string]$destDir) {
  $name = [IO.Path]::GetFileName($src)
  $base = [IO.Path]::GetFileNameWithoutExtension($src)
  $ext  = [IO.Path]::GetExtension($src)
  $dest = Join-Path $destDir $name

  if ($Preview) {
    Write-Host "[PREVIEW] Move: `"$src`" -> `"$dest`""
    return
  }

  if (-not (Test-Path -LiteralPath $dest)) {
    Move-Item -LiteralPath $src -Destination $dest -Force
    return
  }

  $i = 1
  while ($true) {
    $candidate = Join-Path $destDir ("{0} ({1}){2}" -f $base, $i, $ext)
    if (-not (Test-Path -LiteralPath $candidate)) {
      Move-Item -LiteralPath $src -Destination $candidate
      break
    }
    $i++
  }
}

# ==========================
# MAIN
# ==========================
try {
  Backup-DesktopFiles-ToZipDirect
} catch {
  Write-Host "❌ Backup failed: $($_.Exception.Message)"
  exit 9
}

# Sort only FILES on Desktop
$filesToSort = Get-ChildItem -LiteralPath $Desktop -Force | Where-Object { -not $_.PSIsContainer }

$targets = foreach ($f in $filesToSort) {
  if ($ExcludeExactNames -contains $f.Name) { continue }
  $f
}

if (-not $targets -or $targets.Count -eq 0) {
  Write-Host "Nothing to organize."
  exit 0
}

Ensure-Folder $Root

foreach ($f in $targets) {
  $folder = Get-FolderNameFromExt $f.Extension
  $destDir = Join-Path $Root $folder
  Ensure-Folder $destDir
  Move-FileSafely $f.FullName $destDir
}

if ($Preview) {
  Write-Host "PREVIEW finished. (No files were moved.)"
} else {
  Write-Host "DONE."
  Write-Host "ZIP backup: $ZipPath"
  Write-Host "Sorted under: $Root"
}
