param(
  [switch]$Run
)

$Preview = -not $Run

# 콘솔 출력 인코딩(한글 출력 깨짐 방지)
try {
  [Console]::OutputEncoding = [System.Text.UTF8Encoding]::new($false)
  $OutputEncoding = [System.Text.UTF8Encoding]::new($false)
} catch {}

# Desktop path
$Desktop = [Environment]::GetFolderPath("Desktop")

# ZIP Backup settings
$Today = Get-Date -Format "yyyyMMdd"
$ZipName = "바탕화면_백업_$Today.zip"
$ZipPath = Join-Path $Desktop $ZipName

if (Test-Path -LiteralPath $ZipPath) {
  $ts = Get-Date -Format "HHmmss"
  $ZipName = "바탕화면_백업_${Today}_$ts.zip"
  $ZipPath = Join-Path $Desktop $ZipName
}

$TempBackupDir = Join-Path $env:TEMP ("DesktopBackup_" + [guid]::NewGuid())

# Sort root settings
$RootName = "_BY_EXT"
$Root = Join-Path $Desktop $RootName

# Exclude names (extra safety)
$ExcludeExactNames = @(
  $RootName,
  "DesktopSortByExt.ps1",
  "sort_preview.bat",
  "sort_run.bat",
  $ZipName
)

function Ensure-Folder([string]$path) {
  if (-not (Test-Path -LiteralPath $path)) {
    if ($Preview) {
      Write-Host "[PREVIEW] Would create folder: $path"
    } else {
      New-Item -ItemType Directory -Path $path | Out-Null
    }
  }
}

function Backup-Desktop-Zip {
  if ($Preview) {
    Write-Host "[PREVIEW] Would create ZIP backup:"
    Write-Host "          $ZipPath"
    return
  }

  Ensure-Folder $TempBackupDir

  # Copy everything except the zip itself
  Get-ChildItem -LiteralPath $Desktop -Force | Where-Object {
    $_.Name -ne (Split-Path $ZipPath -Leaf)
  } | ForEach-Object {
    Copy-Item -LiteralPath $_.FullName `
      -Destination (Join-Path $TempBackupDir $_.Name) `
      -Recurse -Force
  }

  Compress-Archive -Path "$TempBackupDir\*" -DestinationPath $ZipPath -Force
  Remove-Item -LiteralPath $TempBackupDir -Recurse -Force

  Write-Host "ZIP backup created: $ZipPath"
}

function Get-FolderNameFromExt([string]$ext) {
  if ([string]::IsNullOrWhiteSpace($ext)) { return "_NO_EXT" }
  return $ext.TrimStart('.').ToUpperInvariant()
}

function Move-FileSafely([string]$src, [string]$destDir) {
  $name = [IO.Path]::GetFileName($src)
  $base = [IO.Path]::GetFileNameWithoutExtension($src)
  $ext  = [IO.Path]::GetExtension($src)
  $dest = Join-Path $destDir $name

  # ✅ PREVIEW 모드면 실제 이동 금지
  if ($Preview) {
    Write-Host "[PREVIEW] Move: `"$src`" -> `"$dest`""
    return
  }

  if (-not (Test-Path -LiteralPath $dest)) {
    Move-Item -LiteralPath $src -Destination $dest -Force
    return
  }

  $i = 1
  while ($true) {
    $candidate = Join-Path $destDir ("{0} ({1}){2}" -f $base, $i, $ext)
    if (-not (Test-Path -LiteralPath $candidate)) {
      Move-Item -LiteralPath $src -Destination $candidate
      break
    }
    $i++
  }
}

# ==========================
# MAIN FLOW
# ==========================
Backup-Desktop-Zip

# FILES ONLY
$files = Get-ChildItem -LiteralPath $Desktop -Force | Where-Object { -not $_.PSIsContainer }

$targets = foreach ($f in $files) {
  if ($ExcludeExactNames -contains $f.Name) { continue }
  $f
}

if (-not $targets -or $targets.Count -eq 0) {
  Write-Host "Nothing to organize."
  exit 0
}

Ensure-Folder $Root

foreach ($f in $targets) {
  $folder = Get-FolderNameFromExt $f.Extension
  $destDir = Join-Path $Root $folder
  Ensure-Folder $destDir
  Move-FileSafely $f.FullName $destDir
}

if ($Preview) {
  Write-Host "PREVIEW finished. (No files were moved.)"
} else {
  Write-Host "DONE."
  Write-Host "ZIP backup: $ZipPath"
  Write-Host "Sorted under: $Root"
}
